package gov.va.vamf.scheduling.direct.resource;

import com.agilex.healthcare.directscheduling.domain.BookedAppointments;
import com.agilex.healthcare.directscheduling.utils.DateHelper;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.veteranappointment.resources.AbstractUserResource;
import gov.va.vamf.scheduling.direct.datalayer.appointment.AppointmentDataService;
import gov.va.vamf.scheduling.direct.ical.ICalHelper;
import gov.va.vamf.security.v1.filters.JwtResourceRestricted;
import net.fortuna.ical4j.data.CalendarOutputter;
import net.fortuna.ical4j.model.Calendar;
import net.fortuna.ical4j.model.ValidationException;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;
import javax.ws.rs.core.UriInfo;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Date;

@Path("/ical")
@Component
@Scope("request")
@JwtResourceRestricted
public class DirectSchedulingResource extends AbstractUserResource {
    private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(DirectSchedulingResource.class);

    @Resource
    AppointmentDataService appointmentDataService;

    /**
     * Fetch booked appointments for selected facility from vista for logged in patient/veteran.
     * Service returns collection with booked appointments by "PRIMART CARE" and "OTHER"
     *
     * @param uriInfo
     * @param siteCode
     * @param patientId
     * @param assigningAuthority
     * @param startDateStr
     * @return iCal of booked appointments
     */

    @GET
    @Produces({ "text/calendar" })
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/booked-appointments")
    public Response getBookedAppointmentsIC(
        @Context UriInfo uriInfo,
        @PathParam("site-code") String siteCode,
        @PathParam("assigning-authority") String assigningAuthority,
        @PathParam("patient-id") String patientId,
        @QueryParam("startDate") String startDateStr) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        Date startDate = NullChecker.isNullish(startDateStr) ? new Date() : DateHelper.parseDate(startDateStr);

        BookedAppointments bookedAppointments = appointmentDataService.retrieveBookedAppointments(patientIdentifier,
            siteCode, startDate);

        StreamingOutput stream = generateIcalToStreamingOutput(bookedAppointments);

        return Response.ok(stream).build();
    }

    private StreamingOutput generateIcalToStreamingOutput(final BookedAppointments bookedAppointments) {

        return new StreamingOutput() {
            public void write(OutputStream output) throws IOException, WebApplicationException {
                CalendarOutputter icalOut = new CalendarOutputter();

                try {
                    Calendar ical = ICalHelper.transformAppointmentsToIcal(bookedAppointments);
                    icalOut.output(ical, output);

                } catch (IOException e) {
                    logger.error("IOException in generating iCal output stream", e);
                    throw new WebApplicationException(Response.Status.INTERNAL_SERVER_ERROR);
                } catch (ValidationException e) {
                    logger.error("ValidationException in generating output stream", e);
                    throw new WebApplicationException(Response.Status.INTERNAL_SERVER_ERROR);
                }
            }
        };
    }
}
